// Admin Dashboard JavaScript
class AdminDashboard {
    constructor() {
        this.currentSection = 'ai-personas';
        this.init();
    }

    init() {
        this.setupEventListeners();
        this.showSection(this.currentSection);
    }

    setupEventListeners() {
        // Add Persona Form
        document.getElementById('addPersonaForm').addEventListener('submit', (e) => {
            e.preventDefault();
            this.addPersona();
        });

        // Add Notification Form
        document.getElementById('addNotificationForm').addEventListener('submit', (e) => {
            e.preventDefault();
            this.addNotification();
        });
    }

    showSection(sectionId) {
        // Hide all sections
        document.querySelectorAll('.admin-section').forEach(section => {
            section.style.display = 'none';
        });

        // Show selected section
        document.getElementById(sectionId).style.display = 'block';

        // Update active nav item
        document.querySelectorAll('.admin-nav-item').forEach(item => {
            item.classList.remove('active');
        });
        document.querySelector(`[onclick="showSection('${sectionId}')"]`).classList.add('active');

        this.currentSection = sectionId;
    }

    async addPersona() {
        const form = document.getElementById('addPersonaForm');
        const formData = new FormData(form);

        try {
            const response = await fetch('123admin.php', {
                method: 'POST',
                body: formData
            });
            const data = await response.json();

            if (data.success) {
                this.showNotification(data.message, 'success');
                form.reset();
                // Reload the page to show the new persona
                setTimeout(() => location.reload(), 1500);
            } else {
                this.showNotification(data.message, 'error');
            }
        } catch (error) {
            this.showNotification('Failed to add persona: ' + error.message, 'error');
        }
    }

    async deletePersona(personaId) {
        if (!confirm('Are you sure you want to delete this AI persona? This action cannot be undone.')) {
            return;
        }

        const formData = new FormData();
        formData.append('action', 'delete_persona');
        formData.append('persona_id', personaId);

        try {
            const response = await fetch('123admin.php', {
                method: 'POST',
                body: formData
            });
            const data = await response.json();

            if (data.success) {
                this.showNotification(data.message, 'success');
                // Reload the page to reflect changes
                setTimeout(() => location.reload(), 1500);
            } else {
                this.showNotification(data.message, 'error');
            }
        } catch (error) {
            this.showNotification('Failed to delete persona: ' + error.message, 'error');
        }
    }

    async addNotification() {
        const form = document.getElementById('addNotificationForm');
        const formData = new FormData(form);

        try {
            const response = await fetch('123admin.php', {
                method: 'POST',
                body: formData
            });
            const data = await response.json();

            if (data.success) {
                this.showNotification(data.message, 'success');
                form.reset();
                // Reload the page to show the new notification
                setTimeout(() => location.reload(), 1500);
            } else {
                this.showNotification(data.message, 'error');
            }
        } catch (error) {
            this.showNotification('Failed to add notification: ' + error.message, 'error');
        }
    }

    resetUserPassword(userId, userEmail) {
        document.getElementById('resetPasswordMessage').textContent = 
            `Reset password for user: ${userEmail}`;
        document.getElementById('resetPasswordModal').dataset.userId = userId;
        document.getElementById('resetPasswordModal').style.display = 'flex';
    }

    async confirmResetPassword() {
        const modal = document.getElementById('resetPasswordModal');
        const userId = modal.dataset.userId;
        const newPassword = document.getElementById('new_password').value;

        if (!newPassword) {
            this.showNotification('Please enter a new password', 'error');
            return;
        }

        // In a real application, you would send this to the server
        // For this demo, we'll show a success message
        this.showNotification(`Password reset for user #${userId} would be processed here.`, 'success');
        this.hideModal('resetPasswordModal');
    }

    deleteUser(userId, userEmail) {
        if (!confirm(`Are you sure you want to delete user "${userEmail}"? This action cannot be undone and all their data will be lost.`)) {
            return;
        }

        // In a real application, you would send a delete request to the server
        this.showNotification(`User "${userEmail}" would be deleted here.`, 'success');
    }

    showNotification(message, type = 'info') {
        const notification = document.createElement('div');
        notification.className = `notification ${type}`;
        notification.innerHTML = `
            <div class="notification-icon">
                <i class="mdi mdi-${this.getNotificationIcon(type)}"></i>
            </div>
            <div class="notification-content">
                <p>${message}</p>
            </div>
            <button class="notification-close" onclick="this.parentElement.remove()">
                <i class="mdi mdi-close"></i>
            </button>
        `;
        
        // Create notifications container if it doesn't exist
        let container = document.getElementById('notificationsContainer');
        if (!container) {
            container = document.createElement('div');
            container.className = 'notifications-container';
            container.id = 'notificationsContainer';
            document.body.appendChild(container);
        }
        
        container.appendChild(notification);
        
        // Auto-remove after 5 seconds
        setTimeout(() => {
            if (notification.parentElement) {
                notification.remove();
            }
        }, 5000);
    }

    getNotificationIcon(type) {
        const icons = {
            'success': 'check-circle',
            'error': 'alert-circle',
            'warning': 'alert',
            'info': 'information'
        };
        return icons[type] || 'information';
    }

    hideModal(modalId) {
        document.getElementById(modalId).style.display = 'none';
    }
}

// Global functions for HTML onclick handlers
function showSection(sectionId) {
    adminDashboard.showSection(sectionId);
}

function deletePersona(personaId) {
    adminDashboard.deletePersona(personaId);
}

function resetUserPassword(userId, userEmail) {
    adminDashboard.resetUserPassword(userId, userEmail);
}

function deleteUser(userId, userEmail) {
    adminDashboard.deleteUser(userId, userEmail);
}

function confirmResetPassword() {
    adminDashboard.confirmResetPassword();
}

function hideModal(modalId) {
    adminDashboard.hideModal(modalId);
}

// Initialize admin dashboard
let adminDashboard;
document.addEventListener('DOMContentLoaded', function() {
    adminDashboard = new AdminDashboard();
});